/*
 Small Programmable Power Supply
 Version: 0.2
 Author: Alex from insideGadgets (http://www.insidegadgets.com)
 Created: 8/09/2013
 Last Modified: 22/09/2013
 
 A small programmable power supply based on a P Channel monolithic regulator with an LM358 and digital pot to adjust
 the output voltage with an ATtiny44/84. Uses a 4 digit LED segment display with 3 programmable buttons. 
 
 Voltage input: 12-15V
 Voltage output: 0-12V
 
 */

// ATtiny24/44/84 Pin map
//
//                       +-\/-+
//                 VCC  1|o   |14  GND
//             SDA PB0  2|    |13  PA0 Rotary encoder A
//             SCL PB1  3|    |12  PA1 Rotary encoder B
//           Reset PB3  4|    |11  PA2 
//           Latch PB2  5|    |10  PA3 ADC for 3 Programmable buttons
// ADC for Voltage PA7  6|    |9   PA4 SCK
//              MI PA6  7|    |8   PA5 MO
//                       +----+

#define F_CPU 1000000 // 1 MHz

#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <avr/sleep.h>
#include "setup.c"

#define adcPin 7
#define adcButtons 3

byte readADC = 0; // When to read the ADC
int programButton = 0; // Which programmable button to select
volatile byte dpotValue = 0; // Digital pot value
volatile int dpotDisplaycounter = 0; // Used to show the dpot value for a little while
byte ledsBlink = 0; // Used for blinking LEDs
int blinkCounter = 0; // Blinking LEDs counter

volatile byte numberDisplay[4] = {0, 0, 0, 0}; // Storage of the number to display
volatile byte segmentUpto = 0; // Which segment to light up
volatile byte buttonLeds = 0; // Which button LEDs to light up
volatile byte encoderWaitforlow = false;
volatile byte encoderTimeout = true;

// Shift out the numbers on the segment displays, no need to delay because the timer does that for us
void display_led(byte first, byte second, byte third, byte fourth) {
	if (segmentUpto == 0) {
		latch_low;
		spi_transfer(0x0E | buttonLeds); // First segment GND
		spi_transfer(number_to_led[first]);
		latch_high;
	}
	else if (segmentUpto == 1) {
		latch_low;
		spi_transfer(0x0D | buttonLeds); // Second segment GND
		spi_transfer(number_to_led[second] | (1<<NDP)); // Add dot poin
		latch_high;
	}
	else if (segmentUpto == 2) {
		latch_low;
		spi_transfer(0x0B | buttonLeds);
		spi_transfer(number_to_led[third]);
		latch_high;
	}
	else if (segmentUpto == 3) {
		latch_low;
		spi_transfer(0x07 | buttonLeds);
		spi_transfer(number_to_led[fourth]);
		latch_high;
		segmentUpto = 255; // Reset to zero on next increment
	}
	segmentUpto++;
}

// Store the number to be displayed on the screen
void store_number(double number) {
	if (number > 10) {
		numberDisplay[0] = (int) number / 10;
		numberDisplay[1] = (int) number % 10;
		numberDisplay[2] = (int) (number * 10) % 10;
		numberDisplay[3] = (int) (number * 100) % 10;
	}
	else {
		numberDisplay[0] = BLANK;
		numberDisplay[1] = (int) number;
		numberDisplay[2] = (int) (number * 10) % 10;
		numberDisplay[3] = (int) (number * 100) % 10;
	}
}

int main(void) {
	setup();
	
	while(1) {
		// Read voltage
		if (readADC == 0 && dpotDisplaycounter == 0) {
			int adcValue = analog_read_average(adcPin, REF1_1V); 
			double calculateDouble = (double) ((double) (adcValue) * 1.074) * 0.027015; // Calibrate this for higher accuracy
			store_number(calculateDouble);
		}
		readADC++;
		
		// Read saved buttons: 3.3V - B1 39K - B2 10K - B3 39K - B4 100K and 10K to GND
		int adcValue = analog_read(adcButtons, REF1_1V); 
		
		// First saved button ~673mV
		if (adcValue >= 596 && adcValue <= 656) {
			programButton = 0;
			dpotValue = eeprom_read_byte((uint8_t*) programButton);
			buttonLeds = LED1;
			_delay_ms(250); // Wait a little while for button to be released
		}
		// Second saved button ~559mV
		else if (adcValue >= 490 && adcValue <= 550) {
			programButton = 1;
			dpotValue = eeprom_read_byte((uint8_t*) programButton);
			buttonLeds = LED2;
			_delay_ms(250);
		}
		// Third saved button ~336mV
		else if (adcValue >= 282 && adcValue <= 342) {
			programButton = 2;
			dpotValue = eeprom_read_byte((uint8_t*) programButton);
			buttonLeds = LED3;
			_delay_ms(250);
		}
		// Forth saved button ~166mV
		else if (adcValue >= 124 && adcValue <= 184) {
			blinkCounter = 1000; // Blink LED when saving setting
			ledsBlink = buttonLeds;
			eeprom_write_byte((uint8_t*) programButton, dpotValue);
		}
		
		// Debug: Display digital pot value
		//numberDisplay[1] = (dpotValue/100);
		//numberDisplay[2] = ((dpotValue/10) %10);
		//numberDisplay[3] = (dpotValue % 10);
		
		// Write value to digital pot
		soft_i2c_write_byte(pot_address, dpotValue);
		
		// Used for timings
		if (dpotDisplaycounter > 0) {
			dpotDisplaycounter--;
		}
		if (blinkCounter > 0) {
			if (blinkCounter % 128 == 0) { // Blink LED
				buttonLeds ^= ledsBlink;
			}
			blinkCounter--;
			
			if (blinkCounter == 0) { // Leave LED lit when finished
				buttonLeds |= ledsBlink;
			}
		}
	}
}

ISR(ADC_vect) { }

// Timer0 overflow after 2ms
ISR(TIM0_OVF_vect) {
	display_led(numberDisplay[0], numberDisplay[1], numberDisplay[2], numberDisplay[3]);
}

// Timer1 compare
ISR(TIM1_COMPA_vect) {
	cbi(TCCR1B, CS10);
	encoderTimeout = true;
}

// Pin interrupt on rotary encoder pins
ISR(PCINT0_vect) {
	// Waiting for either pin to go low
	if (encoderWaitforlow == true && !((PINA & (1<<PA1)) && (PINA & (1<<PA0)))) {
		if ((PINA & (1<<PA1)) && dpotValue < 127) { // See which pin is low
			dpotValue++;
		}
		else if ((PINA & (1<<PA0)) && dpotValue > 0) {
			dpotValue--;
		}
		
		dpotDisplaycounter = 400;
		double dpotTemp = ((3.3 / ((dpotValue * 0.078) + 10)) * (dpotValue * 0.078)) * 7.76;
		store_number(dpotTemp);
		
		// Wait for 3ms
		OCR1A = 3000;
		TCNT1 = 0;
		sbi(TCCR1B, CS10);
		encoderWaitforlow = false;
		encoderTimeout = false;
	}
	else if (encoderWaitforlow == false && encoderTimeout == true) { // After the 3ms
		if (!((PINA & (1<<PA1)) && (PINA & (1<<PA0)))) { // If either pin is low, wait another 500uS
			OCR1A = 500;
			TCNT1 = 0;
			sbi(TCCR1B, CS10);
		}
		else {
			encoderWaitforlow = true;
		}
	}
}
