/*
 Small Programmable Power Supply
 Version: 0.1
 Author: Alex from insideGadgets (http://www.insidegadgets.com)
 Created: 8/09/2013
 Last Modified: 8/09/2013

 */
 
#ifndef cbi
#define cbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#endif
#ifndef sbi
#define sbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))
#endif

#ifndef boolean
	typedef uint8_t boolean;
#endif
#ifndef bool
	typedef uint8_t bool;
#endif 
#ifndef byte
	typedef uint8_t byte;
#endif

#ifndef NULL
#define NULL ((void *)0)
#endif

#define LOW 0
#define HIGH 1
#define false 0
#define true 1

#define REF1_1V ((1<<REFS1) | (0<<REFS0))
#define REFVCC ((0<<REFS1) | (0<<REFS0))

#define pot_address 0x2F

#define TWI_SDA_PIN PB0
#define TWI_SCL_PIN PB1
#define I2C_DELAY_USEC 4
#define I2C_READ 1
#define I2C_WRITE 0

#define latch PB2
#define latch_high	PORTB |=  (1<<PB2);
#define latch_low		PORTB &= ~(1<<PB2);

// LED A, B, C pin map to 595 Q0-7 pins
#define NA 7
#define NB 0
#define NC 2
#define ND 4
#define NE 5
#define NF 6
#define NG 1
#define NDP 3

// Button LEDs / GNDs
#define LED1 (1<<4)
#define LED2 (1<<5)
#define LED3 (1<<6)

#define BLANK 10

byte voltindex[20][2];

byte number_to_led[13] = {
	(1<<NA) | (1<<NF) | (1<<NE) | (1<<ND) | (1<<NC) | (1<<NB), // 0
	(1<<NB) | (1<<NC), // 1
	(1<<NA) | (1<<NB) | (1<<NG) | (1<<NE) | (1<<ND), // 2
	(1<<NA) | (1<<NB) | (1<<NG) | (1<<NC) | (1<<ND), // 3
	(1<<NF) | (1<<NG) | (1<<NB) | (1<<NC), // 4
	(1<<NA) | (1<<NF) | (1<<NG) | (1<<NC) | (1<<ND), // 5
	(1<<NA) | (1<<NF) | (1<<NG) | (1<<NE) | (1<<ND) | (1<<NC), // 6
	(1<<NA) | (1<<NB) | (1<<NC), // 7
	(1<<NA) | (1<<NB) | (1<<NC) | (1<<ND) | (1<<NE) | (1<<NF) | (1<<NG), // 8
	(1<<NA) | (1<<NF) | (1<<NG) | (1<<NB) | (1<<NC), // 9
	0, // 10
};

// SPI transfer 1 byte and return the result
uint8_t spi_transfer(uint8_t data) {
	USIDR = data;
	USISR = _BV(USIOIF); // clear flag

	while ( (USISR & _BV(USIOIF)) == 0 ) {
		USICR = (1<<USIWM0)|(1<<USICS1)|(1<<USICLK)|(1<<USITC);
	}
	return USIDR;
}

// Initialize SCL/SDA pins and set the bus high
void SoftI2cMasterInit(void) {
	DDRB |= (1<<TWI_SDA_PIN);
	PORTB |= (1<<TWI_SDA_PIN);
	DDRB |= (1<<TWI_SCL_PIN);
	PORTB |= (1<<TWI_SCL_PIN);
}

// De-initialize SCL/SDA pins and set the bus low
void SoftI2cMasterDeInit(void) {
	PORTB &= ~(1<<TWI_SDA_PIN);
	DDRB &= ~(1<<TWI_SDA_PIN);
	PORTB &= ~(1<<TWI_SCL_PIN);
	DDRB &= ~(1<<TWI_SCL_PIN);
}

// Write a byte to I2C
bool SoftI2cMasterWrite(uint8_t data) {
	// Write byte
	for (uint8_t m = 0x80; m != 0; m >>= 1) {
		// Don't change this loop unless you verify the change with a scope
		if (m & data) { 
			PORTB |= (1<<TWI_SDA_PIN); 
		}
		else { 
			PORTB &= ~(1<<TWI_SDA_PIN); 
		}
		PORTB |= (1<<TWI_SCL_PIN);
		_delay_us(I2C_DELAY_USEC);
		PORTB &= ~(1<<TWI_SCL_PIN);
	}
	// get Ack or Nak
	DDRB &= ~(1<<TWI_SDA_PIN);
	// Enable pullup
	PORTB |= (1<<TWI_SDA_PIN);
	PORTB |= (1<<TWI_SCL_PIN);
	uint8_t rtn = bit_is_set(PINB, TWI_SDA_PIN);
	PORTB &= ~(1<<TWI_SCL_PIN);
	DDRB |= (1<<TWI_SDA_PIN);
	PORTB &= ~(1<<TWI_SDA_PIN);
	return rtn == 0;
}

// Issue a start condition
bool SoftI2cMasterStart(uint8_t addressRW) {
	PORTB &= ~(1<<TWI_SDA_PIN);
	_delay_us(I2C_DELAY_USEC);
	PORTB &= ~(1<<TWI_SCL_PIN);
	return SoftI2cMasterWrite(addressRW);
}

// Issue a stop condition
void SoftI2cMasterStop(void) {
	PORTB &= ~(1<<TWI_SDA_PIN);
	_delay_us(I2C_DELAY_USEC);
	PORTB |= (1<<TWI_SCL_PIN);
	_delay_us(I2C_DELAY_USEC);
	PORTB |= (1<<TWI_SDA_PIN);
	_delay_us(I2C_DELAY_USEC);
}

// Write 1 byte without the write address
bool soft_i2c_write_byte(uint8_t deviceAddr, uint8_t writeByte) {
	
	// Issue a start condition, send device address and write direction bit
	if (!SoftI2cMasterStart((deviceAddr<<1) | I2C_WRITE)) return false;

	// Write the byte
	if (!SoftI2cMasterWrite(writeByte)) return false;

	// Issue a stop condition
	SoftI2cMasterStop();
	
	return true;
}

// ADC Read
int analog_read(uint8_t pin, uint8_t reference) {
	uint8_t low, high;
	
	// Does the reference need to be changed?
	if ((ADMUX & reference) == 0) {
		cbi(ADCSRA, ADEN); // Turn off ADC
		ADMUX = (reference) | (pin & 0x3f); // Set the analog pin and reference
		sbi(ADCSRA, ADEN); // Turn on ADC
	}
	else {
		// Set the analog pin without turning off the ADC
		ADMUX = (reference) | (pin & 0x3f);
	}
	
	set_sleep_mode(SLEEP_MODE_ADC); // Set sleep mode ADC
	sleep_mode(); // System sleeps here
	
	// we have to read ADCL first; doing so locks both ADCL
	// and ADCH until ADCH is read.  reading ADCL second would
	// cause the results of each conversion to be discarded,
	// as ADCL and ADCH would be locked when it completed.
	low = ADCL;
	high = ADCH;
	
	// combine the two bytes
	return (high << 8) | low;
}

// ADC Read with averaging of 5 readings
int analog_read_average(uint8_t pin, uint8_t reference) {
	
	int adcreading = 0;
	for (int x = 0; x < 5; x++) {
		adcreading = adcreading + analog_read(pin, reference);
	}
	adcreading = adcreading / 5;
	
	return adcreading;
}

// Setup the ATtiny
void setup(void) {
	
	// Initialise the SPI
	DDRA |= (1<<PA4); // SPI CLK
	DDRA |= (1<<PA5); // SPI DO
	DDRA &= ~(1<<PA6); // SPI DI
	PORTA |= (1<<PA6); // SPI DI
	DDRB |= (1<<latch); // Latch
	
	// Initialise software I2C/TWI
	SoftI2cMasterInit();
	soft_i2c_write_byte(pot_address, 0); // Reset the Dpot
	
	// Set ADC prescale factor to 8
	// 1 MHz / 8 = 125 KHz, inside the desired 50-200 KHz range.
	sbi(ADCSRA, ADPS1);
	sbi(ADCSRA, ADPS0);
	
	// Turn on ADC interrupt
	sbi(ADCSRA, ADIE);
	
	// Turn on ADC
	sbi(ADCSRA, ADEN);
	
	// Turn on Timer0 overflow interrupt
	sbi(TIMSK0, TOIE0);
	
	// Set Timer0 to 256 prescaler
	TCCR0B = (1<<CS01);
	
	// Pin interrupt setup for the rotary encoder
	sbi(GIMSK,PCIE0); // Enable pin change interrupt
	sbi(PCMSK0,PCINT0);
	sbi(PCMSK0,PCINT1);
	
	// Turn on Timer1 interrupt on OCR1A compare
	sbi(TIMSK1, OCIE1A);
	sbi(TCCR1B, WGM12);
	
	// Write 0 to all 255 blocks for the 3 programmable buttons
	for (int x = 0; x < 3; x++) {
		if (eeprom_read_byte((uint8_t*) x) == 255) {
			eeprom_write_byte((uint8_t*) x, 0);
		}
	}
	
	// Turn on interrupt
	sei();
}
