/*
 AVR Soft I2C Master - Example for I2C EEPROMs
 Version: 1.0
 Author: Alex from insideGadgets (www.insidegadgets.com)
 Created: 12/02/2012
 Last Modified: 12/02/2012
 
 Using code from http://forums.adafruit.com/viewtopic.php?f=25&t=13722

 */

#ifndef cbi
#define cbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#endif
#ifndef sbi
#define sbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))
#endif

#ifndef NULL
#define NULL ((void *)0)
#endif

#define LOW 0
#define HIGH 1
#define false 0
#define true 1

#define I2C_DELAY_USEC 4
#define I2C_READ 1
#define I2C_WRITE 0

// Initialize SCL/SDA pins and set the bus high
void SoftI2cMasterInit(void) {
	DDRB |= (1<<TWI_SDA_PIN);
	PORTB |= (1<<TWI_SDA_PIN);
	DDRB |= (1<<TWI_SCL_PIN);
	PORTB |= (1<<TWI_SCL_PIN);
}

// De-initialize SCL/SDA pins and set the bus low
void SoftI2cMasterDeInit(void) {
	PORTB &= ~(1<<TWI_SDA_PIN);
	DDRB &= ~(1<<TWI_SDA_PIN);
	PORTB &= ~(1<<TWI_SCL_PIN);
	DDRB &= ~(1<<TWI_SCL_PIN);
}

// Read a byte from I2C and send Ack if more reads follow else Nak to terminate read
uint8_t SoftI2cMasterRead(uint8_t last) {
	uint8_t b = 0;
	// Make sure pull-up enabled
	PORTB |= (1<<TWI_SDA_PIN);
	DDRB &= ~(1<<TWI_SDA_PIN);
	// Read byte
	for (uint8_t i = 0; i < 8; i++) {
		// Don't change this loop unless you verify the change with a scope
		b <<= 1;
		_delay_us(I2C_DELAY_USEC);
		PORTB |= (1<<TWI_SCL_PIN);
		if (bit_is_set(PINB, TWI_SDA_PIN)) b |= 1;
		PORTB &= ~(1<<TWI_SCL_PIN);
	}
	// Send Ack or Nak
	DDRB |= (1<<TWI_SDA_PIN);
	if (last) { 
		PORTB |= (1<<TWI_SDA_PIN); 
	}
	else { 
		PORTB &= ~(1<<TWI_SDA_PIN);
	}  
	PORTB |= (1<<TWI_SCL_PIN);
	_delay_us(I2C_DELAY_USEC);
	PORTB &= ~(1<<TWI_SCL_PIN);
	PORTB &= ~(1<<TWI_SDA_PIN);
	return b;
}

// Write a byte to I2C
uint8_t SoftI2cMasterWrite(uint8_t data) {
	// Write byte
	for (uint8_t m = 0x80; m != 0; m >>= 1) {
		// Don't change this loop unless you verify the change with a scope
		if (m & data) { 
			PORTB |= (1<<TWI_SDA_PIN); 
		}
		else { 
			PORTB &= ~(1<<TWI_SDA_PIN); 
		}
		PORTB |= (1<<TWI_SCL_PIN);
		_delay_us(I2C_DELAY_USEC);
		PORTB &= ~(1<<TWI_SCL_PIN);
	}
	// get Ack or Nak
	DDRB &= ~(1<<TWI_SDA_PIN);
	// Enable pullup
	PORTB |= (1<<TWI_SDA_PIN);
	PORTB |= (1<<TWI_SCL_PIN);
	uint8_t rtn = bit_is_set(PINB, TWI_SDA_PIN);
	PORTB &= ~(1<<TWI_SCL_PIN);
	DDRB |= (1<<TWI_SDA_PIN);
	PORTB &= ~(1<<TWI_SDA_PIN);
	return rtn == 0;
}

// Issue a start condition
uint8_t SoftI2cMasterStart(uint8_t addressRW) {
	PORTB &= ~(1<<TWI_SDA_PIN);
	_delay_us(I2C_DELAY_USEC);
	PORTB &= ~(1<<TWI_SCL_PIN);
	return SoftI2cMasterWrite(addressRW);
}

// Issue a restart condition
uint8_t SoftI2cMasterRestart(uint8_t addressRW) {
	PORTB |= (1<<TWI_SDA_PIN);
	PORTB |= (1<<TWI_SCL_PIN);
	_delay_us(I2C_DELAY_USEC);
	return SoftI2cMasterStart(addressRW);
}

// Issue a stop condition
void SoftI2cMasterStop(void) {
	PORTB &= ~(1<<TWI_SDA_PIN);
	_delay_us(I2C_DELAY_USEC);
	PORTB |= (1<<TWI_SCL_PIN);
	_delay_us(I2C_DELAY_USEC);
	PORTB |= (1<<TWI_SDA_PIN);
	_delay_us(I2C_DELAY_USEC);
}

// Write 1 byte
uint8_t soft_i2c_write_byte(uint8_t deviceAddr, uint8_t writeAddress, uint8_t writeByte) {
	
	// Issue a start condition, send device address and write direction bit
	if (!SoftI2cMasterStart((deviceAddr<<1) | I2C_WRITE)) return false;
	
	// Address
	if (!SoftI2cMasterWrite(writeAddress)) return false;
	
	// Write the byte
	if (!SoftI2cMasterWrite(writeByte)) return false;

	// Issue a stop condition
	SoftI2cMasterStop();
	
	return true;
}
