/*
 10x 8x8 LED Clock
 Version: 1.0
 Author: Alex from insideGadgets (http://www.insidegadgets.com)
 Created: 11/03/2015
 Last Modified: 25/06/2015
 
 Uses an Microchip MCP7940M RTC with 10x 8x8 LED matrices to display the time
 Adjust time, trim and timer by buttons
 
 */

// ATmega328 Pin Map
// 
//                 +-\/-+
//     Reset PC6  1|o   |28  PC5 SCL
//  ULN COL8 PD0  2|    |27  PC4 SDA
//  ULN COL7 PD1  3|    |26  PC3 
//  ULN COL6 PD2  4|    |25  PC2 Latch
//  ULN COL5 PD3  5|    |24  PC1 Data
//  ULN COL4 PD4  6|    |23  PC0 Clock
//           VCC  7|    |22  GND
//           GND  8|    |21  AREF
//           PB6  9|    |20  AVCC
//           PB7 10|    |19  PB5 SCK
//  ULN COL3 PD5 11|    |18  PB4 MO AM/PM
//  ULN COL2 PD6 12|    |17  PB3 MI Seconds
//  ULN COL1 PD7 13|    |16  PB2 Minutes
//           PB0 14|    |15  PB1 Hours
//                 +----+

#define F_CPU 8000000

#define RTC_ADDRESS 0x6F 
#define TWI_SDA_PIN PC4
#define TWI_SCL_PIN PC5

#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <avr/pgmspace.h>
#include <avr/sleep.h>
#include "setup.c"

uint8_t LEDs[8][88];

// Processed values
uint8_t hoursTen = 0;
uint8_t hoursOne = 0;
uint8_t minutesTen = 0;
uint8_t minutesOne = 0;
uint8_t secondsTen = 0;
uint8_t secondsOne = 0;
uint8_t timePm = 0;

// Read out values
volatile uint8_t seconds = 0;
volatile uint8_t minutes = 0;
volatile uint8_t hours = 0;

uint8_t rtcCal = 128;
uint8_t resetDisplay = false;

// Shift out to 74HC595 shift registers
void shift_out(uint8_t data) {
	for (int8_t x = 7; x >= 0; x--) {
		if (data & (1<<x)) {
			PORTC |= (1<<PC1);
		}
		else {
			PORTC &= ~(1<<PC1);
		}
		PORTC |= (1<<PC0); // CLK
		PORTC &= ~(1<<PC0);
	}
}

// Clear all LEDs
void clear_LEDs(void) {
	for (uint8_t x = 0; x < 8; x++) {
		for (uint8_t y = 0; y < 88; y++) {
			LEDs[x][y] = 0;
		}
	}
}

// Light up all LEDs
void light_LEDs(void) {
	for (int8_t x = 0; x < 8; x++) {
		for (uint8_t matrix = 0; matrix < 10; matrix++) {
			uint8_t bitsToSend = 0;
			for (uint8_t y = 0; y < 8; y++) {
				if (LEDs[y][x+(matrix*8)] == 1) {
				  bitsToSend |= (1<<y);
				}
			}
			shift_out(bitsToSend);
		}
		
		PORTD = 0; // Turn all columns off
		PORTC |= (1<<PC2); // Latch
		PORTC &= ~(1<<PC2); // Latch
		PORTD = (1<<x);
		
		_delay_us(250);
	}
	
	PORTD = 0; // Turn all columns off
	PORTC |= (1<<PC2); // Latch
	PORTC &= ~(1<<PC2); // Latch
}

void modify_leds(uint8_t position, uint8_t charLocation) {
	position = (position * 8) - 1;
	
	// Append
	for (uint8_t row = 0; row < 8; row++) {
		uint8_t data = (uint8_t) pgm_read_byte(&ledLetters[charLocation][row]);
		
		for (uint8_t column = 0; column < 8; column++) {
			if (data & (1<<column)) {
				LEDs[row][position-column] = 1;
			}
			else {
				LEDs[row][position-column] = 0;
			}
		}
	}	
}

int main(void) {
	setup();
	clear_LEDs();
	
	modify_leds(3, COLON);
	modify_leds(6, COLON);
	modify_leds(9, A);
	modify_leds(10, M);
	
	// Start
	SoftI2cMasterAddress(RTC_ADDRESS, 0, NOREPEATSTART);
	SoftI2cMasterWrite(0x80);
	SoftI2cMasterStop();
	
	// Am/pm mode
	SoftI2cMasterAddress(RTC_ADDRESS, 2, NOREPEATSTART);
	SoftI2cMasterWrite(0x40);
	SoftI2cMasterStop();
	
	while (1) {
		SoftI2cMasterAddress(RTC_ADDRESS, 0, REPEATSTART);
		seconds = SoftI2cMasterRead(0);
		minutes = SoftI2cMasterRead(0);
		hours = SoftI2cMasterRead(1);
		SoftI2cMasterStop();
		
		// Check if changed
		if (hoursTen != ((hours & 0x10) >> 4) + 30) {
			hoursTen = ((hours & 0x10) >> 4) + 30;
			modify_leds(1, hoursTen);
		}
		if (hoursOne != (hours & 0x0F) + 30) {
			hoursOne = (hours & 0x0F) + 30;
			modify_leds(2, hoursOne);
		}
		if (minutesTen != ((minutes & 0x7F) >> 4) + 30) {
			minutesTen = ((minutes & 0x7F) >> 4) + 30;
			modify_leds(4, minutesTen);
		}
		if (minutesOne != (minutes & 0x0F) + 30) {
			minutesOne = (minutes & 0x0F) + 30;
			modify_leds(5, minutesOne);
		}
		if (secondsTen != ((seconds & 0x7F) >> 4) + 30) {
			secondsTen = ((seconds & 0x7F) >> 4) + 30;
			modify_leds(7, secondsTen);
		}
		if (secondsOne != (seconds & 0x0F) + 30) {
			secondsOne = (seconds & 0x0F) + 30;
			modify_leds(8, secondsOne);
		}
		if (timePm != ((hours & 0x20) >> 5)) {
			timePm = (hours & 0x20) >> 5;
			if (timePm == 1) {
				modify_leds(9, P);
				modify_leds(10, M);
			}
			else {
				modify_leds(9, A);
				modify_leds(10, M);
			}
		}
		if (resetDisplay == true) {
			modify_leds(1, hoursTen);
			modify_leds(2, hoursOne);
			modify_leds(4, minutesTen);
			modify_leds(5, minutesOne);
			modify_leds(7, secondsTen);
			modify_leds(8, secondsOne);
			modify_leds(3, COLON);
			modify_leds(6, COLON);
			
			if (timePm == 1) {
				modify_leds(9, P);
				modify_leds(10, M);
			}
			else {
				modify_leds(9, A);
				modify_leds(10, M);
			}
			
			resetDisplay = false;
		}	
		
		// Button detection
		if ((PINB & (1<<PB4 | 1<<PB1)) == (1<<PB4 | 1<<PB1)) {
			rtcCal++;
			
			SoftI2cMasterAddress(RTC_ADDRESS, 8, NOREPEATSTART);
			SoftI2cMasterWrite(rtcCal);
			SoftI2cMasterStop();
			
			modify_leds(1, C);
			modify_leds(2, A);
			modify_leds(3, L);
			
			modify_leds(4, (rtcCal / 100) + 30);
			modify_leds(5, ((rtcCal / 10) % 10) + 30);
			modify_leds(6, (rtcCal % 10) + 30);
			
			modify_leds(7, BK);
			modify_leds(8, BK);
			modify_leds(9, BK);
			modify_leds(10, BK);
			
			_delay_ms(200);
			resetDisplay = true;
		}
		else if ((PINB & (1<<PB4 | 1<<PB2)) == (1<<PB4 | 1<<PB2)) {
			rtcCal--;
			
			SoftI2cMasterAddress(RTC_ADDRESS, 8, NOREPEATSTART);
			SoftI2cMasterWrite(rtcCal);
			SoftI2cMasterStop();
			
			modify_leds(1, C);
			modify_leds(2, A);
			modify_leds(3, L);
			
			modify_leds(4, (rtcCal / 100) + 30);
			modify_leds(5, ((rtcCal / 10) % 10) + 30);
			modify_leds(6, (rtcCal % 10) + 30);
			
			modify_leds(7, BK);
			modify_leds(8, BK);
			modify_leds(9, BK);
			modify_leds(10, BK);
			
			_delay_ms(200);
			resetDisplay = true;
		}
		
		// Timer changes
		else if ((PINB & (1<<PB3 | 1<<PB1)) == (1<<PB3 | 1<<PB1)) {
			OCR0A++;
			
			modify_leds(1, D);
			modify_leds(2, E);
			modify_leds(3, L);
			
			modify_leds(4, (OCR0A / 100) + 30);
			modify_leds(5, ((OCR0A / 10) % 10) + 30);
			modify_leds(6, (OCR0A % 10) + 30);
			
			modify_leds(7, BK);
			modify_leds(8, BK);
			modify_leds(9, BK);
			modify_leds(10, BK);
			
			_delay_ms(200);
			resetDisplay = true;
		}
		else if ((PINB & (1<<PB3 | 1<<PB2)) == (1<<PB3 | 1<<PB2)) {
			OCR0A--;
			
			modify_leds(1, D);
			modify_leds(2, E);
			modify_leds(3, L);
			
			modify_leds(4, (OCR0A / 100) + 30);
			modify_leds(5, ((OCR0A / 10) % 10) + 30);
			modify_leds(6, (OCR0A % 10) + 30);
			
			modify_leds(7, BK);
			modify_leds(8, BK);
			modify_leds(9, BK);
			modify_leds(10, BK);
			
			_delay_ms(200);
			resetDisplay = true;
		}
		
		// Changes to hours, minutes, seconds
		else if (PINB & (1<<PB1)) {
			if ((hours & 0x1F) < 9) {
				hours++;
			}
			else if ((hours & 0x1F) == 9) {
				hours = 0x40 | 1<<4;
			}
			else if ((hours & 0x1F) == 16) {
				hours = 0x40 | 1<<4 | 1<<0;
			}
			else if ((hours & 0x1F) == 17) {
				hours = 0x40 | 1<<4 | 1<<1;
			}
			else {
				hours = 0x40;
			}
			
			SoftI2cMasterAddress(RTC_ADDRESS, 2, NOREPEATSTART);
			SoftI2cMasterWrite(hours);
			SoftI2cMasterStop();
			
			// Re-start RTC
			SoftI2cMasterAddress(RTC_ADDRESS, 0, NOREPEATSTART);
			SoftI2cMasterWrite(seconds | 0x80);
			SoftI2cMasterStop();
		}
		else if (PINB & (1<<PB2)) {
			if ((minutes & 0x0F) < 9) {
				minutes++;
			}
			else {
				uint8_t highMin = (minutes & 0x70) >> 4;
				if (highMin < 5) {
					minutes &= ~(0x70);
					minutes |= (++highMin)<<4;
				}
				else {
					minutes &= ~(0x70);
				}
				
				minutes &= ~(0x0F);
			}
			SoftI2cMasterAddress(RTC_ADDRESS, 1, NOREPEATSTART);
			SoftI2cMasterWrite(minutes);
			SoftI2cMasterStop();
			
			// Re-start RTC
			SoftI2cMasterAddress(RTC_ADDRESS, 0, NOREPEATSTART);
			SoftI2cMasterWrite(seconds | 0x80);
			SoftI2cMasterStop();
		}
		else if (PINB & (1<<PB3)) {
			if ((seconds & 0x0F) < 9) {
				seconds++;
			}
			else {
				uint8_t highSec = (seconds & 0x70) >> 4;
				if (highSec < 5) {
					seconds &= ~(0x70);
					seconds |= (++highSec)<<4;
				}
				else {
					seconds &= ~(0x70);
				}
				
				seconds &= ~(0x0F);
			}
			SoftI2cMasterAddress(RTC_ADDRESS, 0, NOREPEATSTART);
			SoftI2cMasterWrite(seconds);
			SoftI2cMasterStop();
			
			// Re-start RTC
			SoftI2cMasterAddress(RTC_ADDRESS, 0, NOREPEATSTART);
			SoftI2cMasterWrite(seconds | 0x80);
			SoftI2cMasterStop();
		}
		else if (PINB & (1<<PB4)) {
			if (timePm == 0) {
				hours |= 1<<5;
			}
			else {
				hours &= ~(1<<5);
			}
			
			SoftI2cMasterAddress(RTC_ADDRESS, 2, NOREPEATSTART);
			SoftI2cMasterWrite(hours);
			SoftI2cMasterStop();
			
			// Re-start RTC
			SoftI2cMasterAddress(RTC_ADDRESS, 0, NOREPEATSTART);
			SoftI2cMasterWrite(seconds | 0x80);
			SoftI2cMasterStop();
		}
		
		_delay_ms(50);
	}
}

ISR(TIMER0_COMPA_vect) {
	light_LEDs();
}
