/*
 Wireless Gameboy/GBA Controller - Super Nintendo RX
 Version: 1.0
 Author: Alex from insideGadgets (http://www.insidegadgets.com)
 Created: 3/10/2018
 Last Modified: 1/01/2019
 
 Use any Gameboy / GBA which supports a Gameboy or GBA cartridge such as the Gameboy, GBC, GBP, GBA, GBA SP, 
 DS, DS Lite and GB Micro as a Wireless game controller.
 
 
 Program the ATtiny441/841 to use a 16MHz crystal
 avrdude -p ATtiny441 -c usbasp -U lfuse:w:0xff:m -U hfuse:w:0xdf:m
 
*/

// ATtiny441/841 Pin map
//
//                     +-\/-+
//               VCC  1|o   |14  GND
//    16MHz XTAL PB0  2|    |13  PA0 Clock
//    16MHz XTAL PB1  3|    |12  PA1 Data
//         Reset PB3  4|    |11  PA2 LED
//         Latch PB2  5|    |10  PA3 nRF24L01 CE
//  nRF24L01 CSN PA7  6|    |9   PA4 nRF24L01 SCK
//   nRF24L01 DI PA6  7|    |8   PA5 nRF24L01 DO
//                     +----+


#define F_CPU 16000000

#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/sleep.h>
#include <math.h>
#include <util/delay.h>
#include "setup.c"
#include "mirf.c"


uint8_t modeFound = 0;
uint8_t kpData = 0;
uint8_t kpDataHigh = 0;
uint16_t volatile kpOutput = 0xFFFF;
uint8_t firstPacket = 1;
uint8_t kpOutputMapYB = 0;
uint8_t kpOutputMapRtoA = 0;
uint8_t kpOutputMapLtoA = 0;

int main (void) {
	setup();
	
	// Wait for latch high from SNES to sync up 
	while (!(PINB & (1<<LATCH))); // Wait until high
	_delay_us(500);
	
	// Turn on pin interrupt on latch pin
	PCMSK1 |= (1<<PCINT10);
	GIMSK |= (1<<PCIE1);
	
	// Check for new nRF settings for 0.5 seconds
	check_for_new_settings(); 
	mirf_config();
	
	// Check for GB/GBA mode
	check_for_gb_or_gba_mode();
	
	RX_POWERUP;
	mirf_flush_rx_tx(); // Flush TX/RX
	mirf_CE_hi; // Start listening
	_delay_ms(50);
	
	while(1) {
		if (mirf_status() & (1<<RX_DR)) { // Packet received
			PORTA |= (1<<LED);
			mirf_CE_lo; // Stop listening
			
			mirf_CSN_lo; // Pull down chip select
			spi_transfer(R_RX_PAYLOAD); // Send cmd to read rx payload
			
			if (modeFound == GBA_MODE) { // Load 2 bytes if in GBA mode
				kpDataHigh = spi_transfer(0);
				kpData = spi_transfer(0);
			}
			else {
				kpData = spi_transfer(0); // Read data
			}
			
			mirf_CSN_hi; // Pull up chip select
			mirf_config_register(STATUS,(1<<RX_DR)); // Reset status register
			
			mirf_flush_rx_tx(); // Flush TX/RX
			mirf_CE_hi; // Start listening
			PORTA &= ~(1<<LED);
			
			// The first packet received triggers special modes
			if (firstPacket == 1) {
				firstPacket = 0;
				
				// A held down, map the Gameboy A button to the SNES Y button and Gameboy B button to the SNES A button
				if (kpData & 0x01) { // A
					kpOutputMapYB = 1;
				}
				// L held down, map GBA L button to the SNES A button
				if (kpDataHigh & 0x01) {
					kpOutputMapLtoA = 1;
				}
				// R held down, map GBA R button to the SNES A button
				if (kpDataHigh & 0x02) {
					kpOutputMapRtoA = 1;
				}
			}
			
			
			// Check for key presses
			kpOutput = 0xFFFF;
			if (kpData & 0x01) { // A
				if (kpOutputMapYB == 1) {
					kpOutput &= ~(1<<15);
				}
				else {
					kpOutput &= ~(1<<7);
				}
			}
			if (kpData & 0x02) { // B
				if (kpOutputMapYB == 1) {
					kpOutput &= ~(1<<14);
				}
				else {
					kpOutput &= ~(1<<15);
				}
			}
			if (kpData & 0x04) { // Right
				kpOutput &= ~(1<<8);
			}
			if (kpData & 0x08) { // Left
				kpOutput &= ~(1<<9);
			}
			if (kpData & 0x10) { // Up
				kpOutput &= ~(1<<11);
			}
			if (kpData & 0x20) { // Down
				kpOutput &= ~(1<<10);
			}
			if (kpData & 0x40) { // Select
				kpOutput &= ~(1<<13);
			}
			if (kpData & 0x80) { // Start
				kpOutput &= ~(1<<12);
			}
			if (kpDataHigh & 0x01) { // L (for GBA)
				if (kpOutputMapLtoA == 1) {
					kpOutput &= ~(1<<7);
				}
				else {
					kpOutput &= ~(1<<5);
				}
			}
			if (kpDataHigh & 0x02) { // R (for GBA)
				if (kpOutputMapRtoA == 1) {
					kpOutput &= ~(1<<7);
				}
				else {
					kpOutput &= ~(1<<4);
				}
			}
		}
		_delay_us(250);
		
	}
	
	return 0;
}

// We woke up because SNES toggled the latch pin
ISR(PCINT1_vect) {
	if (PINB & (1<<LATCH)) { // Latch high
		PORTA &= ~(1<<LED);
		while (PINB & (1<<LATCH)); // High, wait for it to go low
		
		// 16 bits of data to write depending on what keys were pressed
		for (uint16_t x = 0x8000; x != 0; x >>= 1) { // Loop 8 times
			if (kpOutput & x) {
				PORTA |= (1<<DATA);
			}
			else {
				PORTA &= ~(1<<DATA);
			}
			
			while (PINA & (1<<CLK));
			while (!(PINA & (1<<CLK)));
		}
		
		PORTA &= ~(1<<DATA);
	}
}
