/*
 Wireless Gameboy/GBA Controller - Super Nintendo RX
 Version: 1.0
 Author: Alex from insideGadgets (http://www.insidegadgets.com)
 Created: 3/10/2018
 Last Modified: 1/01/2019
 
 */
 
/*	Copyright (c) 2007 Stefan Engelke <mbox@stefanengelke.de>
	
	Permission is hereby granted, free of charge, to any person 
	obtaining a copy of this software and associated documentation 
	files (the "Software"), to deal in the Software without 
	restriction, including without limitation the rights to use, copy, 
	modify, merge, publish, distribute, sublicense, and/or sell copies 
	of the Software, and to permit persons to whom the Software is 
	furnished to do so, subject to the following conditions:
	
	The above copyright notice and this permission notice shall be 
	included in all copies or substantial portions of the Software.
	
	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
	EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
	MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
	NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
	HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
	WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
	DEALINGS IN THE SOFTWARE.
	
	$Id$
*/

#include "mirf.h"
#include "nRF24L01.h"
#include <avr/io.h>
#include <avr/interrupt.h>

// Defines for setting the MiRF registers for transmitting or receiving mode
#define TX_POWERUP mirf_config_register(CONFIG, mirf_CONFIG | ( (1<<PWR_UP) | (0<<PRIM_RX) ) )
#define RX_POWERUP mirf_config_register(CONFIG, mirf_CONFIG | ( (1<<PWR_UP) | (1<<PRIM_RX) ) )
#define POWERDOWN mirf_config_register(CONFIG, mirf_CONFIG | ( (0<<PWR_UP) ) )

extern uint8_t modeFound;

// Initializes pins ans interrupt to communicate with the MiRF module
// Should be called in the early initializing phase at startup.
void mirf_init(void) {
	// Define CSN and CE as Output and set them to default
	DDRA |= (1<<CE);
	DDRA |= (1<<CSN);
	mirf_CE_lo;
	mirf_CSN_hi;
}

// Set registers in the nRF module
void mirf_config(void) {
	// Set RF channel	
	mirf_config_register(RF_CH, nrfChannel);
	
	// Set length of incoming payload 
	mirf_config_register(RX_PW_P0, mirf_PAYLOAD);
	
	// Disable auto acknowledgement
	mirf_config_register(EN_AA, 0<<ENAA_P0);
	
	// Disable auto retransmit delay
	mirf_config_register(SETUP_RETR, 0);
	
	// Set RADDR and TADDR
	mirf_write_register(RX_ADDR_P0, nrfRx, 5);
	mirf_write_register(TX_ADDR, TADDR, 5);
	
	// Enable RX_ADDR_P0 address matching
	mirf_config_register(EN_RXADDR, (1<<ERX_P0));
}

// Set registers in the nRF module for updating the RX settings
void mirf_config_setting_update(void) {
	// Set RF channel	
	mirf_config_register(RF_CH, mirf_CH_SETTING_UPDATE);
	
	// Set length of incoming payload 
	mirf_config_register(RX_PW_P0, mirf_PAYLOAD_SETTING_UPDATE);
	
	// Set RADDR
	mirf_write_register(RX_ADDR_P0, RADDR_SETTING_UPDATE, 5);
}

// Flush RX and TX FIFO
void mirf_flush_rx_tx(void) {
	mirf_CSN_lo; // Pull down chip select
	spi_transfer(FLUSH_RX); // Flush RX
	mirf_CSN_hi; // Pull up chip select

	mirf_CSN_lo; // Pull down chip select
	spi_transfer(FLUSH_TX);  // Write cmd to flush tx fifo
	mirf_CSN_hi; // Pull up chip select
}

// Read the status register
uint8_t mirf_status(void) {
	mirf_CSN_lo; // Pull down chip select
	spi_transfer(R_REGISTER | (REGISTER_MASK & STATUS));
	uint8_t status = spi_transfer(NOP); // Read status register
	mirf_CSN_hi; // Pull up chip select
	return status;
}

// Checks if data is available for reading
uint8_t mirf_data_ready(void) {
	mirf_CSN_lo; // Pull down chip select
	spi_transfer(R_REGISTER | (REGISTER_MASK & STATUS));
	uint8_t status = spi_transfer(NOP); // Read status register
	mirf_CSN_hi; // Pull up chip select
	return status & (1<<RX_DR);
}

// Checks if MAX_RT has been reached
uint8_t mirf_max_rt_reached(void) {
	mirf_CSN_lo; // Pull down chip select
	spi_transfer(R_REGISTER | (REGISTER_MASK & STATUS));
	uint8_t status = spi_transfer(NOP); // Read status register
	mirf_CSN_hi; // Pull up chip select
	return status & (1<<MAX_RT);
}

// Reads mirf_PAYLOAD bytes into data array
void mirf_get_data(uint8_t *data) {
	mirf_CSN_lo; // Pull down chip select
	spi_transfer(R_RX_PAYLOAD); // Send cmd to read rx payload
	spi_read_data(data, mirf_PAYLOAD); // Read payload
	mirf_CSN_hi; // Pull up chip select
	mirf_config_register(STATUS,(1<<RX_DR)); // Reset status register
}

// Write one byte into the MiRF register
void mirf_config_register(uint8_t reg, uint8_t value) {
	mirf_CSN_lo;
	spi_transfer(W_REGISTER | (REGISTER_MASK & reg));
	spi_transfer(value);
	mirf_CSN_hi;
}

// Reads an array of bytes from the MiRF registers.
void mirf_read_register(uint8_t reg, uint8_t *value, uint8_t len) {
	mirf_CSN_lo;
	spi_transfer(R_REGISTER | (REGISTER_MASK & reg));
	spi_read_data(value, len);
	mirf_CSN_hi;
}

// Writes an array of bytes into the MiRF register
void mirf_write_register(uint8_t reg, uint8_t *value, uint8_t len) {
	mirf_CSN_lo;
	spi_transfer(W_REGISTER | (REGISTER_MASK & reg));
	spi_write_data(value, len);
	mirf_CSN_hi;
}

// Check to see if the TX is sending an updated RX channel/address to use
void check_for_new_settings(void) {
	mirf_config_setting_update();
	RX_POWERUP;
	mirf_flush_rx_tx(); // Flush TX/RX
	mirf_CE_hi; // Start listening
	
	// Check for changing of our RX channel/address for 0.5 seconds
	for (uint8_t timer = 0; timer < 10; timer++) {
		if (mirf_status() & (1<<RX_DR)) { 
			mirf_CE_lo; // Stop listening
			
			mirf_CSN_lo; // Pull down chip select
			spi_transfer(R_RX_PAYLOAD); // Send cmd to read rx payload
			nrfChannel = spi_transfer(0); // Read data
			nrfRx[4] = spi_transfer(0);
			mirf_CSN_hi; // Pull up chip select
			mirf_config_register(STATUS,(1<<RX_DR)); // Reset status register
			
			mirf_flush_rx_tx(); // Flush TX/RX
			
			// Update our EEPROM
			eeprom_write_byte((uint8_t*) nrfChannelLocation, nrfChannel);
			eeprom_write_byte((uint8_t*) nrfRxLocation, nrfRx[4]);
			
			_delay_ms(250);
			PORTA |= (1<<LED);
			_delay_ms(250);
			PORTA &= ~(1<<LED);
			
			break;
		}
		
		_delay_ms(50);
	}
	mirf_CE_lo;
}


// Check to see if we receive 1 or 2 bytes in the packet by switching the receive payload size every 500ms
void check_for_gb_or_gba_mode(void) {
	modeFound = 0;
	
	while (modeFound == 0) {
		for (uint8_t payloadSize = 1; payloadSize <= 2; payloadSize++) {
			mirf_config_register(RX_PW_P0, payloadSize);
			
			RX_POWERUP;
			_delay_ms(3);
			mirf_flush_rx_tx(); // Flush TX/RX
			mirf_CE_hi; // Start listening
			
			for (uint8_t timer = 0; timer < 50; timer++) {
				if (mirf_status() & (1<<RX_DR)) { 
					mirf_CE_lo; // Stop listening
					mirf_flush_rx_tx(); // Flush TX/RX (no need to read the packet)
					mirf_config_register(STATUS,(1<<RX_DR)); // Reset status register
					modeFound = payloadSize;
					
					break;
				}
				_delay_ms(10);
			}
			mirf_CE_lo;
			
			PORTA |= (1<<LED);
			_delay_ms(1);
			PORTA &= ~(1<<LED);
			
			if (modeFound >= 1) {
				break;
			}
		}
	}
}